﻿using System;
using System.Diagnostics;
using System.Reflection;
using System.Security;
using System.Threading;
using log4net;
using log4net.Appender;
using log4net.Core;
using log4net.Layout;
using log4net.Util;
using SecurityContext = log4net.Core.SecurityContext;

namespace gov.va.med.vbecs.Common.Log
{
    /// <summary>
    /// Writes events to the system event log.
    /// 
    /// </summary>
    /// 
    /// <remarks>
    /// 
    /// <para>
    /// The appender will fail if you try to write using an event source that doesn't exist unless it is running with local administrator privileges.
    ///             See also http://logging.apache.org/log4net/release/faq.html#trouble-EventLog
    /// 
    /// </para>
    /// 
    /// <para>
    /// The <c>EventID</c> of the event log entry can be
    ///             set using the <c>EventID</c> property (<see cref="P:log4net.Core.LoggingEvent.Properties"/>)
    ///             on the <see cref="T:log4net.Core.LoggingEvent"/>.
    /// 
    /// </para>
    /// 
    /// <para>
    /// The <c>Category</c> of the event log entry can be
    ///             set using the <c>Category</c> property (<see cref="P:log4net.Core.LoggingEvent.Properties"/>)
    ///             on the <see cref="T:log4net.Core.LoggingEvent"/>.
    /// 
    /// </para>
    /// 
    /// <para>
    /// There is a limit of 32K characters for an event log message
    /// 
    /// </para>
    /// 
    /// <para>
    /// When configuring the EventLogAppender a mapping can be
    ///             specified to map a logging level to an event log entry type. For example:
    /// 
    /// </para>
    /// 
    /// <code lang="XML">
    /// &lt;mapping&gt;
    ///             	&lt;level value="ERROR" /&gt;
    ///             	&lt;eventLogEntryType value="Error" /&gt;
    ///             &lt;/mapping&gt;
    ///             &lt;mapping&gt;
    ///             	&lt;level value="DEBUG" /&gt;
    ///             	&lt;eventLogEntryType value="Information" /&gt;
    ///             &lt;/mapping&gt;
    /// 
    /// </code>
    /// 
    /// <para>
    /// The Level is the standard log4net logging level and eventLogEntryType can be any value
    ///             from the <see cref="T:System.Diagnostics.EventLogEntryType"/> enum, i.e.:
    /// 
    /// <list type="bullet">
    /// 
    /// <item>
    /// <term>Error</term><description>an error event</description>
    /// </item>
    /// 
    /// <item>
    /// <term>Warning</term><description>a warning event</description>
    /// </item>
    /// 
    /// <item>
    /// <term>Information</term><description>an informational event</description>
    /// </item>
    /// 
    /// </list>
    /// 
    /// </para>
    /// 
    /// </remarks>
    public sealed class EventLogAppender : AppenderSkeleton
    {
        /// <summary>
        /// The fully qualified type of the EventLogAppender class.
        /// 
        /// </summary>
        /// 
        /// <remarks>
        /// Used by the internal logger to record the Type of the
        ///             log message.
        /// 
        /// </remarks>
        private static readonly Type DeclaringType = typeof(EventLogAppender);
        /// <summary>
        /// Mapping from level object to EventLogEntryType
        /// 
        /// </summary>
        private readonly LevelMapping _levelMapping = new LevelMapping();
        /// <summary>
        /// The event ID to use unless one is explicitly specified via the <c>LoggingEvent</c>'s properties.
        /// 
        /// </summary>
        private int _eventId;
        /// <summary>
        /// The event category to use unless one is explicitly specified via the <c>LoggingEvent</c>'s properties.
        /// 
        /// </summary>
        private short _category;
        /// <summary>
        /// The log name is the section in the event logs where the messages
        ///             are stored.
        /// 
        /// </summary>
        private string _logName;
        /// <summary>
        /// Name of the application to use when logging.  This appears in the
        ///             application column of the event log named by <see cref="F:log4net.Appender.EventLogAppender._logName"/>.
        /// 
        /// </summary>
        private string _applicationName;
        /// <summary>
        /// The name of the machine which holds the event log. This is
        ///             currently only allowed to be '.' i.e. the current machine.
        /// 
        /// </summary>
        private readonly string _machineName;

        private readonly ILog _logger = log4net.LogManager.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);

        private string _sourceName;

        /// <summary>
        /// The name of the log where messages will be stored.
        /// 
        /// </summary>
        /// 
        /// <value>
        /// The string name of the log where messages will be stored.
        /// 
        /// </value>
        /// 
        /// <remarks>
        /// 
        /// <para>
        /// This is the name of the log as it appears in the Event Viewer
        ///             tree. The default value is to log into the <c>Application</c>
        ///             log, this is where most applications write their events. However
        ///             if you need a separate log for your application (or applications)
        ///             then you should set the <see cref="P:log4net.Appender.EventLogAppender.LogName"/> appropriately.
        /// </para>
        /// 
        /// <para>
        /// This should not be used to distinguish your event log messages
        ///             from those of other applications, the <see cref="P:log4net.Appender.EventLogAppender.ApplicationName"/>
        ///             property should be used to distinguish events. This property should be
        ///             used to group together events into a single log.
        /// 
        /// </para>
        /// 
        /// </remarks>
        public string LogName
        {
            get
            {
                return _logName;
            }
            set
            {
                _logName = value;
            }
        }

        /// <summary>
        /// Property used to set the Application name.  This appears in the
        ///             event logs when logging.
        /// 
        /// </summary>
        /// 
        /// <value>
        /// The string used to distinguish events from different sources.
        /// 
        /// </value>
        /// 
        /// <remarks>
        /// Sets the event log source property.
        /// 
        /// </remarks>
        public string ApplicationName
        {
            get
            {
                return _applicationName;
            }
            set
            {
                _applicationName = value;
            }
        }

        /// <summary>
        /// Property used to set the Fallback Application name.  This appears in the
        ///             event logs when logging.
        /// 
        /// </summary>
        public string ApplicationNameFallBack { get; set; }

        /// <summary>
        /// This property is used to return the name of the computer to use
        ///             when accessing the event logs.  Currently, this is the current
        ///             computer, denoted by a dot "."
        /// 
        /// </summary>
        /// 
        /// <value>
        /// The string name of the machine holding the event log that
        ///             will be logged into.
        /// 
        /// </value>
        /// 
        /// <remarks>
        /// This property cannot be changed. It is currently set to '.'
        ///             i.e. the local machine. This may be changed in future.
        /// 
        /// </remarks>
        public string MachineName
        {
            get
            {
                return _machineName;
            }
// ReSharper disable once ValueParameterNotUsed
            set
            {
            }
        }

        /// <summary>
        /// Gets or sets the <see cref="P:log4net.Appender.EventLogAppender.SecurityContext"/> used to write to the EventLog.
        /// 
        /// </summary>
        /// 
        /// <value>
        /// The <see cref="P:log4net.Appender.EventLogAppender.SecurityContext"/> used to write to the EventLog.
        /// 
        /// </value>
        /// 
        /// <remarks>
        /// 
        /// <para>
        /// The system security context used to write to the EventLog.
        /// 
        /// </para>
        /// 
        /// <para>
        /// Unless a <see cref="P:log4net.Appender.EventLogAppender.SecurityContext"/> specified here for this appender
        ///             the <see cref="P:log4net.Core.SecurityContextProvider.DefaultProvider"/> is queried for the
        ///             security context to use. The default behavior is to use the security context
        ///             of the current thread.
        /// 
        /// </para>
        /// 
        /// </remarks>
        public SecurityContext SecurityContext { get; set; }

        /// <summary>
        /// Gets or sets the <c>EventId</c> to use unless one is explicitly specified via the <c>LoggingEvent</c>'s properties.
        /// 
        /// </summary>
        /// 
        /// <remarks>
        /// 
        /// <para>
        /// The <c>EventID</c> of the event log entry will normally be
        ///             set using the <c>EventID</c> property (<see cref="P:log4net.Core.LoggingEvent.Properties"/>)
        ///             on the <see cref="T:log4net.Core.LoggingEvent"/>.
        ///             This property provides the fallback value which defaults to 0.
        /// 
        /// </para>
        /// 
        /// </remarks>
        public int EventId
        {
            get
            {
                return _eventId;
            }
            set
            {
                _eventId = value;
            }
        }

        /// <summary>
        /// Gets or sets the <c>Category</c> to use unless one is explicitly specified via the <c>LoggingEvent</c>'s properties.
        /// 
        /// </summary>
        /// 
        /// <remarks>
        /// 
        /// <para>
        /// The <c>Category</c> of the event log entry will normally be
        ///             set using the <c>Category</c> property (<see cref="P:log4net.Core.LoggingEvent.Properties"/>)
        ///             on the <see cref="T:log4net.Core.LoggingEvent"/>.
        ///             This property provides the fallback value which defaults to 0.
        /// 
        /// </para>
        /// 
        /// </remarks>
        public short Category
        {
            get
            {
                return _category;
            }
            set
            {
                _category = value;
            }
        }

        /// <summary>
        /// This appender requires a <see cref="N:log4net.Layout"/> to be set.
        /// 
        /// </summary>
        /// 
        /// <value>
        /// <c>true</c>
        /// </value>
        /// 
        /// <remarks>
        /// 
        /// <para>
        /// This appender requires a <see cref="N:log4net.Layout"/> to be set.
        /// 
        /// </para>
        /// 
        /// </remarks>
        protected override bool RequiresLayout
        {
            get
            {
                return true;
            }
        }

        static EventLogAppender()
        {
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="T:log4net.Appender.EventLogAppender"/> class.
        /// 
        /// </summary>
        /// 
        /// <remarks>
        /// 
        /// <para>
        /// Default constructor.
        /// 
        /// </para>
        /// 
        /// </remarks>
        public EventLogAppender()
        {
            _applicationName = Thread.GetDomain().FriendlyName;
            _logName = "Application";
            _machineName = ".";
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="T:log4net.Appender.EventLogAppender"/> class
        ///             with the specified <see cref="T:log4net.Layout.ILayout"/>.
        /// 
        /// </summary>
        /// <param name="layout">The <see cref="T:log4net.Layout.ILayout"/> to use with this appender.</param>
        /// <remarks>
        /// 
        /// <para>
        /// Obsolete constructor.
        /// 
        /// </para>
        /// 
        /// </remarks>
        [Obsolete("Instead use the default constructor and set the Layout property")]
        public EventLogAppender(ILayout layout)
            : this()
        {
            Layout = layout;
        }

        /// <summary>
        /// Add a mapping of level to <see cref="T:System.Diagnostics.EventLogEntryType"/> - done by the config file
        /// 
        /// </summary>
        /// <param name="mapping">The mapping to add</param>
        /// <remarks>
        /// 
        /// <para>
        /// Add a <see cref="T:log4net.Appender.EventLogAppender.Level2EventLogEntryType"/> mapping to this appender.
        ///             Each mapping defines the event log entry type for a level.
        /// 
        /// </para>
        /// 
        /// </remarks>
        public void AddMapping(Level2EventLogEntryType mapping)
        {
            _levelMapping.Add(mapping);
        }

        /// <summary>
        /// Initialize the appender based on the options set
        /// 
        /// </summary>
        /// 
        /// <remarks>
        /// 
        /// <para>
        /// This is part of the <see cref="T:log4net.Core.IOptionHandler"/> delayed object
        ///             activation scheme. The <see cref="M:log4net.Appender.EventLogAppender.ActivateOptions"/> method must
        ///             be called on this object after the configuration properties have
        ///             been set. Until <see cref="M:log4net.Appender.EventLogAppender.ActivateOptions"/> is called this
        ///             object is in an undefined state and must not be used.
        /// 
        /// </para>
        /// 
        /// <para>
        /// If any of the configuration properties are modified then
        ///             <see cref="M:log4net.Appender.EventLogAppender.ActivateOptions"/> must be called again.
        /// 
        /// </para>
        /// 
        /// </remarks>
        public override void ActivateOptions()
        {
            try
            {
                base.ActivateOptions();
                if (SecurityContext == null)
                    SecurityContext = SecurityContextProvider.DefaultProvider.CreateSecurityContext(this);
                bool flag;
                string logName = null;
                // check is event source exists
                using (SecurityContext.Impersonate(this))
                {
                    flag = EventLog.SourceExists(ApplicationName);
                    if (flag)
                    {
                        _sourceName = ApplicationName;
                        logName = EventLog.LogNameFromSourceName(ApplicationName, MachineName);
                    }
                }
                if (!flag || logName != _logName)
                {
                    LogLog.Warn(DeclaringType,
                        "Event source [" + ApplicationName + "] in log [" + LogName + "] doesn't exist. Try to use fallback event source [" + ApplicationNameFallBack + "]");
                    _logger.Warn("Event source [" + ApplicationName + "] in log [" + LogName + "] doesn't exist. Try to use fallback event source [" + ApplicationNameFallBack + "]");
                    flag = EventLog.SourceExists(ApplicationNameFallBack);
                    if (flag)
                    {
                        _sourceName = ApplicationNameFallBack;
                        logName = EventLog.LogNameFromSourceName(ApplicationNameFallBack, MachineName);
                    }
                }
                if (!flag)
                {
                    LogLog.Error(DeclaringType,
                        "Event source [" + ApplicationName + "] and fallback event source [" + ApplicationNameFallBack +
                        "] don't exist. They must be created by a local administrator.  Will disable EventLogAppender.");
                    ErrorHandler.Error("Event source [" + ApplicationName + "] and fallback event source [" + ApplicationNameFallBack + 
                        "] don't exist. They must be created by a local administrator.  Will disable EventLogAppender.");
                    _logger.Error("Event source [" + ApplicationName + "] and fallback event source [" + ApplicationNameFallBack +
                        "] don't exist. They must be created by a local administrator.  Will disable EventLogAppender.");
                    Threshold = Level.Off;
                    return;
                }
                if (logName != _logName)
                {
                    _logger.Error("Event source [" + ApplicationNameFallBack + "] doesn't exist in log [" + LogName + "]. It is located in log [" + logName + "] instead. Will disable EventLogAppender.");
                    ErrorHandler.Error("Event source [" + ApplicationNameFallBack + "] doesn't exist in log [" + LogName + "]. It is located in log [" + logName + "] instead. Will disable EventLogAppender.");
                    Threshold = Level.Off;
                    return;
                }
                _levelMapping.ActivateOptions();
                if (_sourceName != _applicationName)
                {
                    _logger.Warn("Source [" + _sourceName + "] is registered to log [" + LogName + "]");
                }
                else
                {
                    _logger.Debug("Source [" + _sourceName + "] is registered to log [" + LogName + "]");
                }
                LogLog.Debug(DeclaringType, "Source [" + _sourceName + "] is registered to log [" + LogName + "]");
            }
            catch (SecurityException ex)
            {
                _logger.Error(ex.Message, ex);
                ErrorHandler.Error("Caught a SecurityException trying to access the EventLog.  Most likely the event source " + _applicationName + " doesn't exist and must be created by a local administrator.  Will disable EventLogAppender.  See http://logging.apache.org/log4net/release/faq.html#trouble-EventLog", ex);
                Threshold = Level.Off;
            }
        }

        /// <summary>
        /// This method is called by the <see cref="M:log4net.Appender.AppenderSkeleton.DoAppend(log4net.Core.LoggingEvent)"/>
        ///             method.
        /// 
        /// </summary>
        /// <param name="loggingEvent">the event to log</param>
        /// <remarks>
        /// 
        /// <para>
        /// Writes the event to the system event log using the
        ///             <see cref="P:log4net.Appender.EventLogAppender.ApplicationName"/>.
        /// </para>
        /// 
        /// <para>
        /// If the event has an <c>EventID</c> property (see <see cref="P:log4net.Core.LoggingEvent.Properties"/>)
        ///             set then this integer will be used as the event log event id.
        /// </para>
        /// 
        /// <para>
        /// There is a limit of 32K characters for an event log message
        /// 
        /// </para>
        /// 
        /// </remarks>
        protected override void Append(LoggingEvent loggingEvent)
        {
            var eventId = _eventId;
            var obj1 = loggingEvent.LookupProperty("EventID");
            if (obj1 != null)
            {
                if (obj1 is int)
                {
                    eventId = (int)obj1;
                }
                else
                {
                    var s = obj1 as string ?? obj1.ToString();
                    if (s.Length > 0)
                    {
                        int val;
                        if (SystemInfo.TryParse(s, out val))
                            eventId = val;
                        else
                            ErrorHandler.Error("Unable to parse event ID property [" + s + "].");
                    }
                }
            }
            var category = _category;
            var obj2 = loggingEvent.LookupProperty("Category");
            if (obj2 != null)
            {
                if (obj2 is short)
                {
                    category = (short)obj2;
                }
                else
                {
                    var s = obj2 as string ?? obj2.ToString();
                    if (s.Length > 0)
                    {
                        short val;
                        if (SystemInfo.TryParse(s, out val))
                            category = val;
                        else
                            ErrorHandler.Error("Unable to parse event category property [" + s + "].");
                    }
                }
            }
            try
            {
                var message = RenderLoggingEvent(loggingEvent);
                if (_sourceName != _applicationName)
                    message = _applicationName + ": " + message;
                if (message.Length > 32000)
                    message = message.Substring(0, 32000);
                var entryType = GetEntryType(loggingEvent.Level);
                using (SecurityContext.Impersonate(this))
                    EventLog.WriteEntry(_sourceName, message, entryType, eventId, category);
            }
            catch (Exception ex)
            {
                ErrorHandler.Error("Unable to write to event log [" + _logName + "] using source [" + _sourceName + "]", ex);
            }
        }

        /// <summary>
        /// Get the equivalent <see cref="T:System.Diagnostics.EventLogEntryType"/> for a <see cref="T:log4net.Core.Level"/><paramref name="level"/>
        /// </summary>
        /// <param name="level">the Level to convert to an EventLogEntryType</param>
        /// <returns>
        /// The equivalent <see cref="T:System.Diagnostics.EventLogEntryType"/> for a <see cref="T:log4net.Core.Level"/><paramref name="level"/>
        /// </returns>
        /// 
        /// <remarks>
        /// Because there are fewer applicable <see cref="T:System.Diagnostics.EventLogEntryType"/>
        ///             values to use in logging levels than there are in the
        ///             <see cref="T:log4net.Core.Level"/> this is a one way mapping. There is
        ///             a loss of information during the conversion.
        /// 
        /// </remarks>
        private EventLogEntryType GetEntryType(Level level)
        {
            var eventLogEntryType = _levelMapping.Lookup(level) as Level2EventLogEntryType;
            if (eventLogEntryType != null)
                return eventLogEntryType.EventLogEntryType;
            if (level >= Level.Error)
                return EventLogEntryType.Error;
            return level == Level.Warn ? EventLogEntryType.Warning : EventLogEntryType.Information;
        }

        /// <summary>
        /// A class to act as a mapping between the level that a logging call is made at and
        ///             the color it should be displayed as.
        /// 
        /// </summary>
        /// 
        /// <remarks>
        /// 
        /// <para>
        /// Defines the mapping between a level and its event log entry type.
        /// 
        /// </para>
        /// 
        /// </remarks>
        public class Level2EventLogEntryType : LevelMappingEntry
        {
            /// <summary>
            /// The <see cref="P:log4net.Appender.EventLogAppender.Level2EventLogEntryType.EventLogEntryType"/> for this entry
            /// 
            /// </summary>
            /// 
            /// <remarks>
            /// 
            /// <para>
            /// Required property.
            ///             The <see cref="P:log4net.Appender.EventLogAppender.Level2EventLogEntryType.EventLogEntryType"/> for this entry
            /// 
            /// </para>
            /// 
            /// </remarks>
            public EventLogEntryType EventLogEntryType { get; set; }
        }
    }
}
